// Copyright 2014 Google Inc. All Rights Reserved.

#include "MediaBrowserEndpoint.h"

void MediaBrowserEndpoint::addDiscoveryInfo(ServiceDiscoveryResponse* sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    service->mutable_media_browser_service();
}

int MediaBrowserEndpoint::routeMessage(
        uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    uint8_t* ptr = (uint8_t*)msg->raw() + sizeof(uint16_t);
    size_t len = msg->size() - sizeof(uint16_t);

    switch (type) {
        case MEDIA_ROOT_NODE: {
            MediaRootNode node;
            if (PARSE_PROTO(node, ptr, len)) {
                handleRootNode(node);
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case MEDIA_SOURCE_NODE: {
            MediaSourceNode node;
            if (PARSE_PROTO(node, ptr, len)) {
                handleSourceNode(node);
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case MEDIA_LIST_NODE: {
            MediaListNode node;
            if (PARSE_PROTO(node, ptr, len)) {
                handleListNode(node);
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case MEDIA_SONG_NODE: {
            MediaSongNode node;
            if (PARSE_PROTO(node, ptr, len)) {
                handleSongNode(node);
                ret = STATUS_SUCCESS;
            }
            break;
        }
    }
    return ret;
}

int MediaBrowserEndpoint::mediaGetNode(const string& path, uint32_t start, bool get_album_art) {
    MediaGetNode get_node;
    get_node.set_path(path);
    get_node.set_start(start);
    get_node.set_get_album_art(get_album_art);
    IoBuffer buf;
    mRouter->marshallProto(MEDIA_GET_NODE, get_node, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int MediaBrowserEndpoint::reportAction(
    const string& path, uint32_t action) {
    if (InstrumentClusterInput::InstrumentClusterAction_IsValid(action)) {
      MediaBrowserInput input;
      if (!path.empty()) {
          input.set_path(path);
      }
      input.mutable_input()->set_action((InstrumentClusterInput_InstrumentClusterAction)action);
      IoBuffer buf;
      mRouter->marshallProto(MEDIA_BROWSE_INPUT, input, &buf);
      queueOutgoing(buf.raw(), buf.size());
      return STATUS_SUCCESS;
    }
    return STATUS_INVALID_INPUT;
}

void MediaBrowserEndpoint::handleRootNode(const MediaRootNode& node) {
    if (mCallbacks != NULL) {
        MediaRootNodeStruct node_struct;
        node_struct.path = node.path();
        node_struct.num_sources = node.media_sources_size();
        node_struct.sources = new MediaSourceStruct[node_struct.num_sources];
        for (uint32_t i = 0; i < node_struct.num_sources; ++i) {
            copySourceNode(node.media_sources(i), &node_struct.sources[i]);
        }
        mCallbacks->rootNodeCallback(node_struct);
    }
}

void MediaBrowserEndpoint::handleSourceNode(const MediaSourceNode& node) {
    if (mCallbacks != NULL) {
        MediaSourceNodeStruct node_struct;
        copySourceNode(node.source(), &node_struct.source);
        node_struct.start = node.start();
        node_struct.total = node.total();
        node_struct.num_lists = node.lists_size();
        node_struct.lists = new MediaListStruct[node_struct.num_lists];
        for (uint32_t i = 0; i < node_struct.num_lists; ++i) {
            copyListNode(node.lists(i), &node_struct.lists[i]);
        }
        mCallbacks->sourceNodeCallback(node_struct);
    }
}

void MediaBrowserEndpoint::handleListNode(const MediaListNode& node) {
    if (mCallbacks != NULL) {
        struct MediaListNodeStruct node_struct;
        copyListNode(node.list(), &node_struct.media_list);
        node_struct.start = node.start();
        node_struct.total = node.total();
        node_struct.num_songs = node.songs_size();
        node_struct.songs = new MediaSongStruct[node_struct.num_songs];
        for (uint32_t i = 0; i < node_struct.num_songs; ++i) {
            copySongNode(node.songs(i), &node_struct.songs[i]);
        }
        mCallbacks->listNodeCallback(node_struct);
    }
}

void MediaBrowserEndpoint::handleSongNode(const MediaSongNode& node) {
    if (mCallbacks != NULL) {
        struct MediaSongNodeStruct node_struct;
        copySongNode(node.song(), &node_struct.song);
        node_struct.album_art = node.album_art();
        node_struct.duration_seconds = node.duration_seconds();
        mCallbacks->songNodeCallback(node_struct);
    }
}

void MediaBrowserEndpoint::copySourceNode(const MediaSource& src, struct MediaSourceStruct* dest) {
    if (src.has_path()) {
        dest->path = src.path();
    }
    if (src.has_name()) {
        dest->name = src.name();
    }
    if (src.has_album_art()) {
        dest->album_art = src.album_art();
    }
}

void MediaBrowserEndpoint::copyListNode(const MediaList& src, struct MediaListStruct* dest) {
    dest->path = src.path();
    dest->type = src.type();
    dest->name = src.name();
    dest->album_art = src.album_art();
}

void MediaBrowserEndpoint::copySongNode(const MediaSong& src, struct MediaSongStruct* dest) {
    dest->path = src.path();
    dest->name = src.name();
    dest->artist = src.artist();
    dest->album = src.album();
}
